use std::fmt::Formatter;

use fnv::{FnvHashMap, FnvHashSet};
use serde::{Deserialize, Deserializer, Serialize, Serializer};
use serde::de::{Error, Visitor};

#[derive(Debug, Hash, Eq, PartialEq, Clone)]
pub enum ShieldVarValue{
    Int(i64),
    Bool(bool),
    Str(String)
}

pub type AgentLabelStep = Vec<ShieldVarValue>;
pub type AgentLabelHistory = Vec<AgentLabelStep>;

pub type AgentLabelStepSet = Vec<AgentLabelStep>;

pub type AgentLabelHistorySet = Vec<AgentLabelHistory>;

impl Serialize for ShieldVarValue{
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error> where S: Serializer {
        match self {
            ShieldVarValue::Int(i) => {serializer.serialize_i64(*i)}
            ShieldVarValue::Bool(b) => {serializer.serialize_bool(*b)}
            ShieldVarValue::Str(s) => {serializer.serialize_str(s)}
        }
    }
}

struct ShieldVarValueVisitor;
impl <'de> Visitor<'de> for ShieldVarValueVisitor {
    type Value = ShieldVarValue;

    fn expecting(&self, formatter: &mut Formatter) -> std::fmt::Result {
        formatter.write_str("an integer, boolean, or string")
    }

    fn visit_bool<E>(self, v: bool) -> Result<Self::Value, E> where E: Error {
        Ok(ShieldVarValue::Bool(v))
    }

    fn visit_i64<E>(self, v: i64) -> Result<Self::Value, E> where E: Error {
        Ok(ShieldVarValue::Int(v))
    }

    fn visit_u64<E>(self, v: u64) -> Result<Self::Value, E> where E: Error {
        match i64::try_from(v) {
            Ok(i) => Ok(ShieldVarValue::Int(i)),
            Err(e) => Err(E::custom(e.to_string()))
        }
    }

    fn visit_str<E>(self, v: &str) -> Result<Self::Value, E> where E: Error {
        Ok(ShieldVarValue::Str(v.parse().unwrap()))
    }

    fn visit_string<E>(self, v: String) -> Result<Self::Value, E> where E: Error {
        Ok(ShieldVarValue::Str(v))
    }
}

impl<'de> Deserialize<'de> for ShieldVarValue{
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error> where D: Deserializer<'de> {
        deserializer.deserialize_any(ShieldVarValueVisitor)
    }
}

#[derive(Deserialize, Debug)]
pub struct ActionAndSuccessors{
    pub action: Vec<u32>,
    pub successors: Vec<u32>
}

#[derive(Deserialize, Debug)]
pub struct PartialObsCentralizedShieldState {
    pub observations: Vec<Vec<AgentLabelStep>>,  // For each agent [List of possible observations]
    pub actions: Vec<ActionAndSuccessors>,
    pub initial: bool,
    pub hidden: Vec<FnvHashSet<ShieldVarValue>>
}

#[derive(Deserialize, Debug)]
pub struct PartialObsCentralizedShield {
    pub action_space: Vec<u32>,
    pub shield_states: FnvHashMap<u32, PartialObsCentralizedShieldState>,
    pub obs_names: Vec<Vec<String>>,
    pub hidden_obs_names: Vec<String>,
}


