use std::collections::HashSet;

use fnv::{FnvHashMap, FnvHashSet};
use serde::Deserialize;

use crate::decentralization::sat_shield::StateTo;
use crate::shields::{AgentLabelHistory, AgentLabelStep};
use crate::shields::partial_obs_cent::{AgentLabelHistorySet, PartialObsCentralizedShield, ShieldVarValue};

type StateLabelHistory = Vec<AgentLabelHistory>;
pub type StateHistoryInfo = Vec<AgentHistoryInfo>;

#[derive(Clone, Eq, PartialEq)]
pub struct AgentHistoryInfo {
    pub initial_infos: HashSet<AgentLabelHistory>,
    pub non_init_infos: HashSet<AgentLabelHistory>
}

impl AgentHistoryInfo{
    pub fn merge(&mut self, other: Self) {
        for slhi in other.initial_infos {
            self.initial_infos.insert(slhi);
        }
        for slhi in other.non_init_infos {
            self.non_init_infos.insert(slhi);
        }
        self.non_init_infos.retain(|slhi| !self.initial_infos.contains(slhi));
    }

    fn add_successor_to_all(hs: &HashSet<AgentLabelHistory>, succ_obs_possibilities: &Vec<AgentLabelStep>) -> HashSet<AgentLabelHistory> {
        hs.iter().flat_map(|info| {
            let succs_of_info : Vec<AgentLabelHistory> = succ_obs_possibilities.iter().map(|succ_obs| {
                let mut new_single_initial_info: AgentLabelHistory = info.clone();
                new_single_initial_info.push(succ_obs.clone());
                new_single_initial_info
            }).collect();
            succs_of_info
        }).collect()
    }

    pub fn add_successor_obs(&self, succ_obs_possibilities: &Vec<AgentLabelStep>) -> Self {
        Self {
            initial_infos: Self::add_successor_to_all(&self.initial_infos, succ_obs_possibilities),
            non_init_infos: Self::add_successor_to_all(&self.non_init_infos, succ_obs_possibilities)
        }
    }
}

#[derive(Deserialize, Debug)]
pub struct PartialObsHistCentralizedShieldState {
    pub observations: Vec<AgentLabelHistorySet>,
    pub actions: Vec<Vec<u32>>,
    pub initial: bool,
    pub hidden: Vec<FnvHashSet<ShieldVarValue>>
}

#[derive(Deserialize, Debug)]
pub struct PartialObsHistCentralizedShield {
    pub action_space: Vec<u32>,
    pub shield_states: FnvHashMap<u32, PartialObsHistCentralizedShieldState>,
    pub obs_names: Vec<Vec<String>>,
    pub hidden_obs_names: Vec<String>,
    pub history_len: u8,
}

pub fn iter_label_history(centralized_shield: &PartialObsCentralizedShield, label_histories_of_state: &StateTo<StateHistoryInfo>) -> StateTo<StateHistoryInfo>{
    let mut next_label_histories_of_state = label_histories_of_state.clone();
    for (state_num, state_info) in centralized_shield.shield_states.iter(){
        for action_info in state_info.actions.iter(){
            for successor in action_info.successors.iter(){
                let possible_obs_of_successor = &centralized_shield.shield_states[successor].observations;
                let label_histories_of_current = &label_histories_of_state[state_num];

                let label_histories_of_succesor = next_label_histories_of_state.get_mut(successor).unwrap();
                for (all_possible_obs_of_successor_agent, (label_history_of_agent, label_history_of_successor_agent)) in possible_obs_of_successor.iter().zip(label_histories_of_current.iter().zip(label_histories_of_succesor.iter_mut())) {
                    let new_label_histories_of_successor = label_history_of_agent.add_successor_obs(all_possible_obs_of_successor_agent);
                    label_history_of_successor_agent.merge(new_label_histories_of_successor)
                }
            }
        }
    }

    next_label_histories_of_state
}

fn extract_label_history_from_centralized_shield(centralized_shield: &PartialObsCentralizedShield) -> StateTo<StateHistoryInfo> {
    centralized_shield.shield_states.iter()
        .map(|(snum, sinfo)| {
            let agent_histories = sinfo.observations.iter().map(|agent_possible_observations| {
                let single_step_history_set: HashSet<AgentLabelHistory> = agent_possible_observations.iter().map(|possible_observation| vec![possible_observation.clone()]).collect();

                if sinfo.initial {
                    AgentHistoryInfo {
                        initial_infos: single_step_history_set,
                        non_init_infos: HashSet::new()
                    }
                }   else {
                    AgentHistoryInfo {
                        initial_infos: HashSet::new(),
                        non_init_infos: single_step_history_set
                    }
                }
            }).collect();
            (*snum, agent_histories)
        }).collect()
}

pub fn create_hist_shield_from_label_history(centralized_shield: &PartialObsCentralizedShield, label_histories_of_state: StateTo<StateHistoryInfo>, history_len: u8) -> PartialObsHistCentralizedShield{
    let mut state_mapping: StateTo<PartialObsHistCentralizedShieldState> = FnvHashMap::default();
    let mut new_state_counter = 0u32;

    for (state_num, history_set) in label_histories_of_state {
        let state_in_centralized_shield = &centralized_shield.shield_states[&state_num];
        let actions : Vec<_> = state_in_centralized_shield.actions.iter().map(|a| a.action.clone()).collect();

        let reachable_history_sets = history_set.into_iter().map(|mut agent_history_info| {
            let mut reachable_infos = agent_history_info.initial_infos;
            reachable_infos.extend(
                agent_history_info.non_init_infos.into_iter().filter(|possible_history| {possible_history.len() == (history_len + 1) as usize})
            );
            reachable_infos.into_iter().collect()
        }).collect();


        let this_new_state = new_state_counter;
        new_state_counter += 1;

        state_mapping.insert(this_new_state, PartialObsHistCentralizedShieldState {
            initial: state_in_centralized_shield.initial,
            hidden: state_in_centralized_shield.hidden.clone(),
            actions: actions.clone(),
            observations: reachable_history_sets
        });
    }

    PartialObsHistCentralizedShield {
        shield_states: state_mapping,
        action_space: centralized_shield.action_space.clone(),
        obs_names: centralized_shield.obs_names.clone(),
        hidden_obs_names: centralized_shield.hidden_obs_names.clone(),
        history_len
    }
}

pub fn create_history_centralized_shield(centralized_shield: &PartialObsCentralizedShield, history_len: u8) -> PartialObsHistCentralizedShield {
    let mut state_label_history = extract_label_history_from_centralized_shield(centralized_shield);

    for _ in 0..history_len {
        state_label_history = iter_label_history(centralized_shield, &state_label_history);
    }

    create_hist_shield_from_label_history(centralized_shield, state_label_history, history_len)
}