extern crate kissat;
extern crate serde;

use std::fs::File;
use std::io::{BufReader, BufWriter};
use std::process::exit;
use std::time::Instant;

use clap::{Arg, ArgAction, Command, value_parser};
use crate::decentralization::alt_sat_shield::find_decentralized_shield_sat_alt;

use crate::decentralization::arbitrary_dec_shield::find_decentralized_shield_arbitrary;
use crate::decentralization::sat_shield::find_decentralized_shield_sat;
use crate::shields::{create_history_centralized_shield, DecentralizedShield, PartialObsCentralizedShield};

mod shields;
mod decentralization;


fn main() {
    let args = Command::new("Decentralized Shield Synthesis Tool")
        .arg(Arg::new("base_path"))
        .arg(Arg::new("history_len").short('l').long("history_len").value_parser(value_parser!(u8)).default_value("0"))
        .arg(Arg::new("method").short('m').long("method").default_value("sat"))
        .arg(Arg::new("disable_fullness_check").short('d').long("disable_fullness_check").action(ArgAction::SetTrue))
        .get_matches();

    let path_to_shield_base = args.get_one::<String>("base_path").unwrap();
    let history_len = *args.get_one::<u8>("history_len").unwrap();
    let method = args.get_one::<String>("method").unwrap().to_lowercase();

    let mut path_to_cent_shield = path_to_shield_base.clone();
    path_to_cent_shield.push_str(".shield_pobs_cent");

    let start_time = Instant::now();
    let shield : PartialObsCentralizedShield = serde_json::from_reader(BufReader::new(File::open(&path_to_cent_shield).unwrap())).unwrap();
    println!("{:.2?}: Loaded shield from disk", start_time.elapsed());
    let step_time = Instant::now();
    let hist_shield = create_history_centralized_shield(&shield, history_len);

    println!("{:.2?} Shield processed, attempting to decentralize with method {}", start_time.elapsed(), method);
    let maybe_solution = if method == "sat" {
        find_decentralized_shield_sat(&hist_shield)
    } else if method == "naive" {
        let disable_fullness = *args.get_one::<bool>("disable_fullness_check").unwrap();
        find_decentralized_shield_arbitrary(&hist_shield, disable_fullness)
    } else if method == "alt_sat" {
        find_decentralized_shield_sat_alt(&hist_shield)
    } else {
        panic!("method must be \"naive\" or \"sat\"")
    };
    let proc_dec_time = step_time.elapsed();

    if let Some(solution) = maybe_solution{
        println!("{:.2?} Shield Decentralization Succeeded, writing to file", start_time.elapsed());
        let mut path_to_dec_shield = path_to_shield_base.clone();
        path_to_dec_shield.push_str(".shield_pobs_dec");

        let dec_shield = DecentralizedShield::new(solution, shield.obs_names, history_len);
        serde_json::to_writer_pretty(BufWriter::new(File::create(&path_to_dec_shield).unwrap()), &dec_shield).unwrap();
        println!("{:.2?} Shield written to file", start_time.elapsed());
        println!("Process + dec time {:.2?}", proc_dec_time);
        println!("Load + save time   {:.2?}", start_time.elapsed() - proc_dec_time);
        } else {
        println!("Shield Decentralization Failed, try removing states");
        exit(-1);
    }
}